<?php
require("header.php");
?>

<div class="log-container">
    <!-- 控制卡片 -->
    <div class="log-control-card">
        <div class="log-header">
            <div class="log-icon">
                <i class="fas fa-scroll"></i>
            </div>
            <div class="log-title">日志管理</div>
        </div>
        
        <div class="log-controls">
            <select id="logFileSelect" class="log-select">
                <option value="">加载中...</option>
            </select>
            
            <button id="deleteLogBtn" class="log-btn delete-btn">
                <i class="fas fa-trash"></i> 删除日志
            </button>
        </div>
    </div>
    
    <!-- 日志列表容器 -->
    <div id="logListContainer" class="log-list">
        <div class="log-placeholder">
            <i class="fas fa-file-alt"></i>
            <p>请选择日志文件查看内容</p>
        </div>
    </div>
</div>

<!-- 日志详情弹窗 -->
<div id="logDetailModal" class="log-modal">
    <div class="modal-content">
        <div class="modal-header">
            <h3 id="logDetailTitle">日志详情</h3>
            <button class="modal-close-btn">&times;</button>
        </div>
        <div class="log-detail-content">
            <pre id="logDetailContent"></pre>
        </div>
        <div class="modal-buttons">
            <button class="modal-btn confirm-btn">确定</button>
        </div>
    </div>
</div>

<style>
    /* 日志容器样式 */
    .log-container {
        max-width: 800px;
        margin: 10px auto 20px;
        padding: 15px 20px;
    }
    
    /* 控制卡片样式 */
    .log-control-card {
        background-color: white;
        border-radius: 10px;
        padding: 15px;
        box-shadow: 0 2px 10px rgba(0, 0, 0, 0.05);
        margin-bottom: 15px;
    }
    
    .log-header {
        display: flex;
        align-items: center;
        margin-bottom: 15px;
    }
    
    .log-icon {
        width: 36px;
        height: 36px;
        background-color: #f0f8ff;
        border-radius: 8px;
        display: flex;
        align-items: center;
        justify-content: center;
        margin-right: 12px;
    }
    
    .log-icon i {
        font-size: 16px;
        color: #4da6ff;
    }
    
    .log-title {
        font-size: 17px;
        font-weight: bold;
        flex: 1;
    }
    
    .log-controls {
        display: flex;
        align-items: center;
        gap: 10px;
    }
    
    .log-select {
        flex: 1;
        padding: 8px 12px;
        border: 1px solid #e0e0e0;
        border-radius: 6px;
        font-size: 14px;
        background-color: #f8fafc;
        transition: all 0.2s;
    }
    
    .log-select:focus {
        border-color: #4da6ff;
        outline: none;
        box-shadow: 0 0 0 2px rgba(77, 166, 255, 0.2);
    }
    
    /* 日志按钮样式 */
    .log-btn {
        padding: 8px 12px;
        border: none;
        border-radius: 6px;
        font-size: 13px;
        cursor: pointer;
        display: flex;
        align-items: center;
        gap: 5px;
        transition: all 0.2s;
    }
    
    .delete-btn {
        background-color: #ff6b6b;
        color: white;
    }
    
    .delete-btn:hover {
        background-color: #e05555;
    }
    
    /* 日志列表样式 */
    .log-list {
        background-color: white;
        border-radius: 10px;
        padding: 15px;
        box-shadow: 0 2px 10px rgba(0, 0, 0, 0.05);
        max-height: 600px;
        overflow-y: auto;
    }
    
    .log-placeholder {
        display: flex;
        flex-direction: column;
        align-items: center;
        justify-content: center;
        padding: 40px 0;
        color: #999;
    }
    
    .log-placeholder i {
        font-size: 36px;
        margin-bottom: 12px;
        color: #e0e0e0;
    }
    
    .log-placeholder p {
        margin: 0;
        font-size: 15px;
    }
    
    /* 日志卡片样式 */
    .log-card {
        background-color: white;
        border-radius: 8px;
        padding: 12px 15px;
        margin-bottom: 10px;
        box-shadow: 0 2px 6px rgba(0, 0, 0, 0.04);
        border-left: 4px solid #4da6ff;
        cursor: pointer;
        transition: all 0.2s;
        display: flex;
        align-items: flex-start;
        gap: 12px;
    }
    
    .log-card:hover {
        box-shadow: 0 4px 10px rgba(0, 0, 0, 0.08);
        transform: translateY(-1px);
    }
    
    .log-card-icon {
        width: 24px;
        height: 24px;
        border-radius: 6px;
        display: flex;
        align-items: center;
        justify-content: center;
        flex-shrink: 0;
        margin-top: 2px;
    }
    
    .log-card-icon i {
        font-size: 14px;
        color: white;
    }
    
    .log-card-content {
        flex: 1;
    }
    
    .log-card-time {
        font-size: 12px;
        color: #666;
        margin-bottom: 5px;
    }
    
    .log-card-summary {
        font-size: 14px;
        line-height: 1.5;
    }
    
    /* 弹窗样式 */
    .log-modal {
        display: none;
        position: fixed;
        top: 0;
        left: 0;
        right: 0;
        bottom: 0;
        background-color: rgba(0, 0, 0, 0.5);
        z-index: 2000;
        justify-content: center;
        align-items: center;
    }
    
    .log-modal .modal-content {
        background-color: white;
        padding: 0;
        border-radius: 12px;
        width: 600px;
        max-width: 90%;
        max-height: 80vh;
        display: flex;
        flex-direction: column;
        box-shadow: 0 4px 20px rgba(0, 0, 0, 0.15);
    }
    
    .modal-header {
        display: flex;
        justify-content: space-between;
        align-items: center;
        padding: 16px 20px;
        border-bottom: 1px solid #f0f0f0;
    }
    
    .modal-header h3 {
        margin: 0;
        font-size: 16px;
        color: #333;
    }
    
    .modal-close-btn {
        background: none;
        border: none;
        font-size: 20px;
        color: #999;
        cursor: pointer;
        padding: 0;
        width: 24px;
        height: 24px;
        display: flex;
        align-items: center;
        justify-content: center;
        transition: all 0.2s;
    }
    
    .modal-close-btn:hover {
        color: #666;
        background-color: #f5f5f5;
        border-radius: 4px;
    }
    
    .log-detail-content {
        flex: 1;
        overflow-y: auto;
        padding: 16px 20px;
        background-color: #f8f9fa;
    }
    
    .log-detail-content pre {
        margin: 0;
        white-space: pre-wrap;
        font-family: 'Courier New', Courier, monospace;
        font-size: 13px;
        line-height: 1.4;
        color: #333;
    }
    
    .modal-buttons {
        display: flex;
        justify-content: flex-end;
        padding: 12px 20px;
        border-top: 1px solid #f0f0f0;
    }
    
    .modal-btn {
        padding: 8px 16px;
        border: none;
        border-radius: 6px;
        font-size: 14px;
        cursor: pointer;
        transition: all 0.2s;
    }
    
    .confirm-btn {
        background-color: #e1f0ff;
        color: #4da6ff;
    }
    
    .confirm-btn:hover {
        background-color: #d0e6ff;
    }
    
    /* 自定义滚动条 */
    ::-webkit-scrollbar {
        width: 6px;
        height: 6px;
    }
    
    ::-webkit-scrollbar-track {
        background: #f1f1f1;
        border-radius: 4px;
    }
    
    ::-webkit-scrollbar-thumb {
        background: #c1c1c1;
        border-radius: 4px;
    }
    
    ::-webkit-scrollbar-thumb:hover {
        background: #a8a8a8;
    }
    
    /* 图标类型颜色 */
    .log-icon-group {
        background-color: #4da6ff;
    }
    
    .log-icon-private {
        background-color: #6c5ce7;
    }
    
    .log-icon-join {
        background-color: #00b894;
    }
    
    .log-icon-leave {
        background-color: #ff7675;
    }
    
    .log-icon-default {
        background-color: #636e72;
    }
</style>

<script>
    // 页面加载时初始化
    document.addEventListener('DOMContentLoaded', function() {
        // 加载日志文件列表
        loadLogFiles();
        
        // 绑定事件
        bindEvents();
    });
    
    // 绑定事件
    function bindEvents() {
        // 日志文件选择变化
        document.getElementById('logFileSelect').addEventListener('change', function() {
            const logFile = this.value;
            if (logFile) {
                loadLogContent(logFile);
            } else {
                showPlaceholder();
            }
        });
        
        // 删除日志按钮
        document.getElementById('deleteLogBtn').addEventListener('click', function() {
            const logFile = document.getElementById('logFileSelect').value;
            if (!logFile) {
                alert('请先选择要删除的日志');
                return;
            }
            
            if (confirm(`确定要删除日志 "${logFile}" 吗？`)) {
                deleteLogFile(logFile);
            }
        });
        
        // 日志详情弹窗确定按钮
        document.querySelector('#logDetailModal .confirm-btn').addEventListener('click', function() {
            document.getElementById('logDetailModal').style.display = 'none';
        });
        
        // 弹窗关闭按钮
        document.querySelector('#logDetailModal .modal-close-btn').addEventListener('click', function() {
            document.getElementById('logDetailModal').style.display = 'none';
        });
    }
    
    // 加载日志文件列表
    function loadLogFiles() {
        fetch('api/log.php?type=list')
            .then(response => response.json())
            .then(data => {
                if (data.code === 200) {
                    renderLogFileSelect(data.list);
                    // 默认选择今天的日志
                    const todayLog = new Date().toISOString().slice(0, 10) + '.log';
                    if (data.list.includes(todayLog)) {
                        document.getElementById('logFileSelect').value = todayLog;
                        loadLogContent(todayLog);
                    }
                } else {
                    alert('获取日志列表失败');
                }
            })
            .catch(error => {
                console.error('Error:', error);
                alert('获取日志列表时出错');
            });
    }
    
    // 渲染日志文件选择框
    function renderLogFileSelect(logFiles) {
        const select = document.getElementById('logFileSelect');
        select.innerHTML = '';
        
        if (logFiles.length === 0) {
            select.innerHTML = '<option value="">没有找到日志文件</option>';
            return;
        }
        
        // 按日期倒序排列
        logFiles.sort().reverse();
        
        logFiles.forEach(file => {
            const option = document.createElement('option');
            option.value = file;
            option.textContent = file;
            select.appendChild(option);
        });
    }
    
    // 加载日志内容
    function loadLogContent(logFile) {
        fetch(`api/log.php?type=read&name=${logFile}`)
            .then(response => response.json())
            .then(data => {
                if (data.code === 200) {
                    renderLogList(data.list);
                } else {
                    alert(data.msg || '读取日志内容失败');
                    showPlaceholder();
                }
            })
            .catch(error => {
                console.error('Error:', error);
                alert('读取日志内容时出错');
                showPlaceholder();
            });
    }
    
    // 渲染日志列表
    function renderLogList(logs) {
        const container = document.getElementById('logListContainer');
        
        if (!logs || logs.length === 0) {
            container.innerHTML = `
                <div class="log-placeholder">
                    <i class="fas fa-info-circle"></i>
                    <p>该日志无内容</p>
                </div>
            `;
            return;
        }
        
        let html = '';
        logs.forEach(log => {
            // 解析日志类型
            const logType = getLogType(log.raw);
            const iconInfo = getIconInfo(logType);
            
            html += `
                <div class="log-card" data-time="${log.time}" data-raw="${escapeHtml(log.raw)}">
                    <div class="log-card-icon ${iconInfo.class}">
                        <i class="fas ${iconInfo.icon}"></i>
                    </div>
                    <div class="log-card-content">
                        <div class="log-card-time">${log.time}</div>
                        <div class="log-card-summary">${log.summary || '无摘要'}</div>
                    </div>
                </div>
            `;
        });
        
        container.innerHTML = html;
        
        // 绑定卡片点击事件
        document.querySelectorAll('.log-card').forEach(card => {
            card.addEventListener('click', function() {
                const time = this.dataset.time;
                const raw = this.dataset.raw;
                
                document.getElementById('logDetailTitle').textContent = `日志详情 - ${time}`;
                document.getElementById('logDetailContent').textContent = raw;
                document.getElementById('logDetailModal').style.display = 'flex';
            });
        });
    }
    
    // 获取日志类型
    function getLogType(raw) {
        try {
            const logData = JSON.parse(raw);
            return logData.t || '';
        } catch (e) {
            return '';
        }
    }
    
    // 获取图标信息
    function getIconInfo(logType) {
        switch(logType) {
            case 'GROUP_AT_MESSAGE_CREATE':
                return { class: 'log-icon-group', icon: 'fa-users' };
            case 'C2C_MESSAGE_CREATE':
                return { class: 'log-icon-private', icon: 'fa-user' };
            case 'GROUP_ADD_ROBOT':
                return { class: 'log-icon-join', icon: 'fa-sign-in-alt' };
            case 'GROUP_DEL_ROBOT':
                return { class: 'log-icon-leave', icon: 'fa-sign-out-alt' };
            default:
                return { class: 'log-icon-default', icon: 'fa-info-circle' };
        }
    }
    
    // 显示占位符
    function showPlaceholder() {
        document.getElementById('logListContainer').innerHTML = `
            <div class="log-placeholder">
                <i class="fas fa-file-alt"></i>
                <p>请选择日志查看内容</p>
            </div>
        `;
    }
    
    // 删除日志文件
    function deleteLogFile(logFile) {
        fetch(`api/log.php?type=delete&name=${logFile}`)
            .then(response => response.json())
            .then(data => {
                if (data.code === 200) {
                    alert('日志删除成功');
                    loadLogFiles();
                    showPlaceholder();
                } else {
                    alert(data.msg || '日志删除失败');
                }
            })
            .catch(error => {
                console.error('Error:', error);
                alert('删除日志时出错');
            });
    }
    
    // HTML转义函数
    function escapeHtml(unsafe) {
        return unsafe
            .replace(/&/g, "&amp;")
            .replace(/</g, "&lt;")
            .replace(/>/g, "&gt;")
            .replace(/"/g, "&quot;")
            .replace(/'/g, "&#039;");
    }
</script>

<?php
require("footer.php");
?>